package it.unich.easytelegram;

import java.io.IOException;
import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.invoke.MethodType;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

import org.telegram.telegrambots.bots.TelegramLongPollingBot;
import org.telegram.telegrambots.meta.TelegramBotsApi;
import org.telegram.telegrambots.meta.api.objects.Update;
import org.telegram.telegrambots.updatesreceivers.DefaultBotSession;

public class BotController extends TelegramLongPollingBot {

    private String userName;

    private String token;

    private boolean logging;

    private MethodHandle bot;

    private Map<Long, Chat> chats = new HashMap<>();

    public BotController(MethodHandle bot) throws IOException, NoSuchMethodException {
        super();
        this.bot = bot;
        Properties p = new Properties();
        p.load(BotController.class.getClassLoader().getResourceAsStream("config.properties"));
        this.userName = p.getProperty("easytelegram.username");
        this.token = p.getProperty("easytelegram.token");
        var loggingStr = p.getProperty("easytelegram.logging");
        this.logging = (loggingStr == null) ? true : Boolean.parseBoolean(loggingStr);
    }

    @Override
    public void onUpdateReceived(Update update) {
        // We check if the update has a message and the message has text
        log("Update received");
        if (update.hasMessage() && update.getMessage().hasText()) {
            var message = update.getMessage();
            var chatId = message.getChatId();
            Chat chat;
            chat = chats.get(chatId);
            if (chat == null) {
                chat = new Chat(chatId, this);
                chats.put(chatId, chat);
            }
            chat.processInput(message.getText());
        }
    }

    public MethodHandle getBot() {
        return bot;
    }

    @Override
    public String getBotUsername() {
        return userName;
    }

    @Override
    public String getBotToken() {
        return token;
    }

    void log(String msg) {
        if (logging)
            System.out.println(msg);
    }

    void log(Chat chat, String msg) {
        if (logging)
            System.out.println("chat " + chat.getId() + " : " + msg);
    }

    public static void launch() {
        var caller = StackWalker.getInstance(StackWalker.Option.RETAIN_CLASS_REFERENCE).getCallerClass();
        try {
            var lookup = MethodHandles.publicLookup();
            var method = lookup.findStatic(caller, "run", MethodType.methodType(void.class, Chat.class));
            TelegramBotsApi botsApi = new TelegramBotsApi(DefaultBotSession.class);
            botsApi.registerBot(new BotController(method));
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
